# from fleming.system_test.fle_sys import *

# Imports
import asyncio
import socket
import os
import fnmatch
import json
import time
import configparser
import traceback
import math
import re

from enum import IntEnum

from pylog.pylogger import PyLogger

from datetime import datetime

from py_pli.pylib import VUnits
from py_pli.pylib import send_msg
from py_pli.pylib import GlobalVar

import config_enum.scan_table_enum as scan_table_enum
import config_enum.excitationlight_selector_enum as els_enum
import config_enum.filter_module_slider_enum as fms_enum
import config_enum.bottom_light_director_enum as bld_enum
import config_enum.focus_mover_enum as fm_enum
import config_enum.platedoor_enum as pd_enum
import config_enum.detector_aperture_slider_enum as das_enum

PATH, SCRIPT = os.path.split(__file__)
BASE_NAME = SCRIPT.split('.')[0]
ADJUSTMENT_PATH = '/home/pkiuser/.config/pyrunner/'

# Logging and Messages
logger = PyLogger.logger
error_msg = f"Error Predefined Tasks in {BASE_NAME}."
start_msg = f"Start Predefined Task {BASE_NAME}."
info_msg = f"Information from {BASE_NAME}."
stop_msg = f"Stopped Predefined Task {BASE_NAME}."
complete_msg = f"Completed Predefined Task {BASE_NAME}."

# mover-dict in order of homing
all_mover = {
    'fm': VUnits.instance.hal.focusMover,
    'st': VUnits.instance.hal.scan_table,
    'pd': VUnits.instance.hal.plateDoor,
    'fms': VUnits.instance.hal.filterModuleSlider,
    'das1': VUnits.instance.hal.detectorApertureSlider1,
    'das2': VUnits.instance.hal.detectorApertureSlider2,
    'els': VUnits.instance.hal.excitationLightSelector,
    'bld': VUnits.instance.hal.bottomLightDirector,
    'sd': VUnits.instance.hal.doorLatch,
}


# sample for predefined positions:
# els.GotoPosition(fms_enum.Positions.FixMirrorPosition_c)

# _check : Go/NoGo
# _test  : Test returns result value
# _scan  : Scan returns detailed data



async def mover_position_test(mover = 'pd', profile = 1, num_of_cycles = 1):
    # checks saved positions
    # Mover:
    # fm: FocusMover
    # fms: FilterModuleSlider
    # pd: PlateDoor
    # das1: DetectorApertureSlider1
    # das2: DetectorApertureSlider2
    # els: ExcitationLightSelector: Producing high step errors end_pos > 355 (>5000msteps)
    # bld: BottomLightDirector: Producing high step errors end_pos > 345 (>5000msteps)

    # Tested 2022-8-02/kstruebing on Newport

    TEST_NAME = 'mover_position_check'

    msg = f'{start_msg}{TEST_NAME}'
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))
    
    t_mover = all_mover[mover]
    try:
        GlobalVar.set_stop_gc(False)
        cycles = int(num_of_cycles) if num_of_cycles else 1
        # 
        if mover == 'fm':
            saved_positions = t_mover.configManager.get_section(fm_enum.Positions)
        elif mover == 'fms':
            saved_positions = t_mover.configManager.get_section(fms_enum.Positions)
        elif mover == 'pd':
            saved_positions = t_mover.configManager.get_section(pd_enum.Positions)
        elif mover == 'das1':
            saved_positions = t_mover.configManager.get_section(das_enum.Positions)
        elif mover == 'das2':
            saved_positions = t_mover.configManager.get_section(das_enum.Positions)
        elif mover == 'els':
            saved_positions = t_mover.configManager.get_section(els_enum.Positions)
        elif mover == 'bld':
            saved_positions = t_mover.configManager.get_section(bld_enum.Positions)
 
        if not saved_positions:
            msg = f"{error_msg}{TEST_NAME} : No saved position was found in the mover configuration file"
            logger.error(msg)
            await send_msg(json.dumps({"result": msg}))
            GlobalVar.set_stop_gc(True)
            return 

        await t_mover.Home()
        await t_mover.UseProfile(profile)

        for i in range(cycles):
            for name, position in saved_positions.items():
                if GlobalVar.get_stop_gc():
                    msg = f"{stop_msg}.{TEST_NAME} : Stop Button pressed by User"
                    logger.error(msg)
                    await send_msg(json.dumps({"result": msg}))
                    return 

                msg = f"{info_msg}{TEST_NAME} : Cycle {i}: Moving to Position {name} => {position}"
                logger.info(msg)
                await send_msg(json.dumps({"result": msg}))
                await t_mover.Move(position)
                await asyncio.sleep(0.5)

        await t_mover.Home()
        step_error = t_mover.Mover.LastHomeStepErrors
        msg = f"{info_msg}{TEST_NAME} : Result: Step Errors({mover}): {step_error}"
        logger.info(msg)
        await send_msg(json.dumps({"result": msg}))
        return(step_error)

    except Exception as ex:
        msg = f"{error_msg}{TEST_NAME} -> {traceback.format_exc()}"
        logger.error(msg)
        await send_msg(json.dumps({"result": msg}))
        GlobalVar.set_stop_gc(True)  # Stop script execution
        return (msg)

async def release_plate_check(cycles = 1, position = 307.0, delay = 0.1):
    TEST_NAME = "release_plate_check"

    msg = f'{start_msg}{TEST_NAME}'
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))

    release_position_y = position
    release_position_x = 1
    
    try:
        GlobalVar.set_stop_gc(False)
        plate_door = all_mover["pd"]
        await plate_door.UseProfile(1)
        await plate_door.Open()
        scan_table = all_mover["st"]
        await scan_table.Home()
        await scan_table.UseProfile(1)
        x = release_position_x
        y = release_position_y
        for i in range(cycles):
            await scan_table.Move(x, y)
            time.sleep(delay)
            await scan_table.Move(x, y - 10.0)
            time.sleep(delay)
        step_errors = await scan_table.Home()
        

    except Exception as ex:
        msg = f"{error_msg}{TEST_NAME} : {str(ex)}: {traceback.format_exc()}"
        logger.error(msg)
        await send_msg(json.dumps({"result": msg}))
        GlobalVar.set_stop_gc(True)
    
    msg = f'{complete_msg}{TEST_NAME} : finished, Step Errors = {step_errors}' 
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))
    
    return(msg)
        
async def scan_table_well_test(cycles = 1, profile = 3, plate_format="384", detector="fbdtop", delay_ms=10):
    """
    parameters:
    cycles:             no of cycles to run each plate
    profile:            ScanTable Motor Profile No. 
    plate_format(str):  default is 384, allowed are "96", "384", "1536"
    detector(str):      default is "fbdtop", allowed are "us", "fbdtop", "fbdbottom"
    delay_ms(int):      delay between well to well movement, default is 300s

    workflow:
    Device is initialized, scan table is homed, moves to well A1 .
    Now it iterates over the defined plate (waits for delay_ms ms in each well).
    Scan table home, return step errors.
    
    Tested 2022-8-02/kstruebing on Newport
    """
        
    # Scriptname and Start Message
    TEST_NAME = "st_well_scan"
    msg = f'{start_msg}{TEST_NAME}'
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))
    
    try:
        GlobalVar.set_stop_gc(False)
        scan_table = all_mover["st"]


        #set parameters:
        mover_profile = int(profile) if profile else 1
        plate_type = str(plate_format) if plate_format else "384"
        detector_pos = str(detector) if detector else "us"
        delay = int(delay_ms)/1000 if delay_ms else 300/1000

        #set plate type
        # cols, rows swapped
        if plate_type == "384":
            scan_table.SetPlateType("'384 OptiPlate (Black)'")
            rows = 16
            cols = 24
        if plate_type == "96":
            scan_table.SetPlateType("'96 OptiPlate (Black)'")
            rows = 8
            cols = 12
        if plate_type == "1536":
            scan_table.SetPlateType("'1536 General (White)'")
            rows = 32
            cols = 48

        #set detector meas pos
        if detector_pos == "us":
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.USLum_TopLeftCorner)
        if detector_pos == "fbdtop":
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.FBDTop_TopLeftCorner)
        if detector_pos == "fbdbottom":
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.FBDBottom_TopLeftCorner)

        #logger.info("gc_predefined_tasks> Current plate: {scan_table.currentPlate}")

        await scan_table.Home()
        await scan_table.UseProfile(1)
        await scan_table.MoveToWell(1, 1)
        await scan_table.UseProfile(mover_profile)
        
        for i in range(cycles):
            start_row = time.time()
            for row in range(1, rows+1, 1):
                start_col = time.time()
                if row%2==1:
                    for col in range(1, cols+1):
                        await scan_table.MoveToWell(col, row)
                        time.sleep(delay)
                        msg = f'{info_msg}{TEST_NAME} : cycle {i}: well x, y, {col}, {row}'
                        await send_msg(json.dumps({"result": msg}))
                else:
                    for col in range(cols, 0, -1):
                        await scan_table.MoveToWell(col, row) 
                        time.sleep(delay)
                        msg = f'{info_msg}{TEST_NAME} : cycle {i}: well x, y, {col}, {row}'
                        await send_msg(json.dumps({"result": msg}))
                t_col = (time.time() - start_col) / cols
            t_row = (time.time() - start_row) / rows
        step_errors = await scan_table.Home()

        msg = f'{complete_msg}{TEST_NAME} : step errors = {step_errors}, time per well = {t_col:.3f}s, time per row = {t_row:.3f}s'
        await send_msg(json.dumps({"result": msg}))

        return(msg)

    except Exception as ex:
        msg = f"{error_msg}{TEST_NAME} : {str(ex)}: {traceback.format_exc()}"
        logger.error(msg)
        await send_msg(json.dumps({"result": msg}))
        GlobalVar.set_stop_gc(True)
        return

async def scan_table_otf_test(profile = 4, plate_format="384", detector="fbdtop", delay_ms=300):
    """
    parameters:
    plate_format(str): default is 384, allowed are "96", "384", "1536"
    detector(str):     default is "us", allowed are "us", "fbdtop", "fbdbottom"
    delay_ms(int):      default is 300s

    workflow:
    Device is initialized, scan table is homed, moves to well A1 .
    Now it iterates over the defined plate (waits for delay_ms ms in each well).
    Scan table home, return step errors.
    """
    TEST_NAME = "scan_table_otf_scan"
    msg = f'{start_msg}{TEST_NAME}'
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))

    try:
        GlobalVar.set_stop_gc(False)

        scan_table = all_mover["st"]

        #set parameters:
        mover_profile = int(profile) if profile else 1
        plate_type = str(plate_format) if plate_format else "384"
        detector_pos = str(detector) if detector else "us"
        delay = int(delay_ms)/1000 if delay_ms else 300/1000

        #set plate type
        # cols, rows swapped
        if plate_type == "384":
            scan_table.SetPlateType("'384 OptiPlate (Black)'")
            rows = 16
            cols = 24
        if plate_type == "96":
            scan_table.SetPlateType("'96 OptiPlate (Black)'")
            rows = 8
            cols = 12
        if plate_type == "1536":
            scan_table.SetPlateType("'1536 General (White)'")
            rows = 32
            cols = 48

        #set detector meas pos
        if detector_pos == "us":
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.USLum_TopLeftCorner)
        if detector_pos == "fbdtop":
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.FBDTop_TopLeftCorner)
        if detector_pos == "fbdbottom":
            scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.FBDBottom_TopLeftCorner)

        logger.info("gc_predefined_tasks> Current plate: {scan_table.currentPlate}")

        await scan_table.Home()
        await scan_table.UseProfile(1)
        await scan_table.MoveToWell(1, 1)
        await scan_table.UseProfile(mover_profile)

        start = time.time()
        for row in range(1, rows+1, 1):
            if row%2==1:
                for col in range(1, cols+1, cols):
                    await scan_table.MoveToWell(col, row)
                    time.sleep(delay)
                    msg = f'{info_msg}{TEST_NAME} : well x, y, {col}, {row}'
                    await send_msg(json.dumps({"result": msg}))
            else:
                for col in range(cols, 0, -cols):
                    await scan_table.MoveToWell(col, row)    
                    msg = f'{info_msg}{TEST_NAME} : well x, y, {col}, {row}'
                    await send_msg(json.dumps({"result": msg}))
        t_sum = (time.time() - start)
        step_errors = await scan_table.Home()

        msg = f'{complete_msg}{TEST_NAME} : scan table step errors = {step_errors}, Time = {t_sum:.3f}s'

        return (msg)

    except Exception as ex:
        msg = f"{error_msg}{TEST_NAME} : {str(ex)}: {traceback.format_exc()}"
        logger.error(msg)
        GlobalVar.set_stop_gc(True)

        return(msg)
  
async def scan_table_check(profile = 1, diagonal_mode = 1, num_of_cycles=1):
    """
    Iterate through ScanTable positions and check step errors.
    [x] tested 2022-08-01/kstruebing/Newport
    """
    TEST_NAME = 'scan_table_check'
    msg = f'{start_msg}{TEST_NAME}'
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))

    scan_table = all_mover["st"]
    try:
        GlobalVar.set_stop_gc(False)
        cycles = int(num_of_cycles) if num_of_cycles else 1

        await scan_table.Home()
        await scan_table.UseProfile(profile)
        if diagonal_mode == 1:
            scan_table.CoreXY.cartesianMode = False
        for i in range(cycles):
            if GlobalVar.get_stop_gc():
                msg = f'{stop_msg}{TEST_NAME} : Stop Button pressed by User'
                logger.error(msg)
                await send_msg(json.dumps({"result": msg}))
                scan_table.CoreXY.cartesianMode = True
                return (msg)

            msg = f'{info_msg}{TEST_NAME} : cycle: {i + 1}'
            logger.info(msg)
            await send_msg(json.dumps({"result": msg}))
            
            # Pattern to check all moving directions
            # |\\
            await scan_table.Move(0, 176)   # front, right
            await scan_table.Move(170, 0)   # rear, left
            await scan_table.Move(0, 176)   # front, right
            await asyncio.sleep(0.5)        # Pause
            # -//
            await scan_table.Move(170, 176) # front, left
            await scan_table.Move(0, 0)     # rear, right
            await scan_table.Move(170, 176) # front, left
            await asyncio.sleep(0.5)        # Pause
            # |-
            await scan_table.Move(170, 0)   # rear, left
            await scan_table.Move(0, 0)     # rear, right
            await asyncio.sleep(0.5)        # Pause
            # -|-|
            await scan_table.Move(170, 0)   # rear, left
            await scan_table.Move(170, 176) # front, left
            await scan_table.Move(0, 176)   # front, right
            await scan_table.Move(0, 0)     # rear, right
            await asyncio.sleep(0.5)        # Pause

        await scan_table.Home()
        step_error = scan_table.CoreXY.LastHomeStepErrors
        msg = f'{complete_msg}{TEST_NAME} : Step Errors(scan_table): {step_error}' 
        logger.info(msg)
        # await send_msg(json.dumps({"result": msg}))
        scan_table.CoreXY.cartesianMode = True
        return (msg) 

    except Exception as ex:
        msg = f"{error_msg}{TEST_NAME} : {str(ex)}: {traceback.format_exc()}"
        logger.error(msg)
        await send_msg(json.dumps({"result": msg}))
        scan_table.CoreXY.cartesianMode = True
        GlobalVar.set_stop_gc(True)

        return(msg)

async def scan_table_shake_test(shaking_mode = 0, duration_s = 1, speed_rpm = 30.0, diameter_mm = 1.0, ):
    # 2022-03-04 16:08:49/ created: kstruebing
    # 2022-04-19 FLE-10767
    # Test of shaking function
    # Estimated Limits:
    # diameter = 0.1 mm {-}> speed ≥ 90 RPM
    # speed = 30 RPM -> diameter ≥ 0.9 mm
    # speed = 30 RPM -> diameter ≥ 0.9 mm

    TEST_NAME = 'Shaking Test'
    msg = f'{start_msg}{TEST_NAME}'
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))

    result = ""
    diameter = float(diameter_mm) if diameter_mm else 1.0
    speed = float(speed_rpm) if speed_rpm else 30.0
    mode = int(shaking_mode) if shaking_mode else 0
    duration = int(duration_s) if duration_s else 1
    estimated_duration = float(duration) - float(duration)/10.0
    
    try:
        await enable_shaking(mode, duration, speed, diameter)
        await wait_shaking_finished(estimated_duration)
    
    except Exception as ex:
        result = f"{complete_msg}{TEST_NAME}: Failed: {ex}"

    result = f"{complete_msg}{TEST_NAME}: OK"
    return result

async def sliding_door_test(cycles, period_s):
    # Test of sliding door locking mechanism
    # No sensors, return value, just an acoustic test
    #  
    TEST_NAME = 'sliding_door_test'
    sliding_door = all_mover['sd']
    
    for i in range(cycles):
        sliding_door.Unlock()
        msg = f'{complete_msg}{TEST_NAME} : Cycle {i}: Sliding Door unlocked' 
        await send_msg(json.dumps({"result": msg}))

        await asyncio.sleep(period_s)
        sliding_door.Lock()
        msg = f'{complete_msg}{TEST_NAME} : Cycle {i}: Sliding Door locked' 
        await send_msg(json.dumps({"result": msg}))

        await asyncio.sleep(period_s)

    msg = f'{complete_msg}{TEST_NAME} : Sliding Door Test done.' 
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))
    return()

async def scan_table_parameter_test(num_of_cycles = 1, profile = 1):
    # 2022-08-04/kstruebing
    # Test of scan_table acceleration parameter
    # Runs movertest with different acceleration/deceleration settings and check step-errors
    # cycles: cycles to run movertest
    # profile: profile which will be modified

    TEST_NAME = 'scan_table_parameter_test'
    ACCELERATION = (20.0, 100.0, 500.0, 1000.0, 2000.0, 5000.0, 10000.0)
    #ACCELERATION = (2000, 5000)
    cycles = int(num_of_cycles) if num_of_cycles else 1
    profile_i = int(profile) if profile else 1
    msg = f'{start_msg}{TEST_NAME}'
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))

    scan_table = all_mover["st"]
    #await scan_table.UseProfile(profile)

    for i in range(cycles):
        for a in ACCELERATION:
            await scan_table.CoreXY.SetProfile(handle = profile_i, 
                                        speed = 600.0, 
                                        accelDV = a, 
                                        decelDV = a, 
                                        drivePower = 60, 
                                        holdPower = 20, 
                                        drivePowerHoldTime = 1000, 
                                        drivePowerFallTime = 1000) 
            await scan_table.UseProfile(profile)
            # Move square and home
            try:
                await scan_table.Move(5, 176)   # front, right
                await scan_table.Move(170, 176)   # front, left
                await scan_table.Move(170, 5)   # rear, left
                await scan_table.Move(5, 5)   # rear, right
            except Exception as ex:
                await scan_table.Home()
                step_error = scan_table.CoreXY.LastHomeStepErrors
                msg = f'cycle {i + 1}: acc/dec: {a}; step_error:{step_error}'
                await send_msg(json.dumps({"result": msg}))
                msg = f"{complete_msg}{TEST_NAME}: Failed: {ex}"
                return(msg)
                
            await scan_table.Home()
            step_error = scan_table.CoreXY.LastHomeStepErrors
            msg = f'cycle {i + 1}: acc/dec: {a}; step_error:{step_error}'
            await send_msg(json.dumps({"result": msg}))
    msg = f"{complete_msg}{TEST_NAME}: OK"
    logger.info(msg)
    return(msg)

async def scan_table_acc_test(profile = 1):
    # 2022-08-05/kstruebing
    # Test of scan_table acceleration test
    # Runs movertest with different acceleration/deceleration settings and check step-errors
    # cycles: cycles to run movertest
    # profile: profile which will be modified

    TEST_NAME = 'scan_table_parameter_test'
    ACCELERATION = (5000.0, 6000.0, 7000.0, 8000.0, 9000.0, 100000.0)
    profile_i = int(profile) if profile else 1
    msg = f'{start_msg}{TEST_NAME}'
    logger.info(msg)
    await send_msg(json.dumps({"result": msg}))

    scan_table = all_mover["st"]
    #await scan_table.UseProfile(profile)

    for a in ACCELERATION:
        await scan_table.CoreXY.SetProfile(handle = profile_i, 
                                    speed = 600.0, 
                                    accelDV = a, 
                                    decelDV = a, 
                                    drivePower = 60, 
                                    holdPower = 20, 
                                    drivePowerHoldTime = 1000, 
                                    drivePowerFallTime = 1000) 
        await scan_table.UseProfile(profile)
        # Move square and home
        try:
            await scan_table.Move(5, 176)   # front, right
            await scan_table.Move(170, 176)   # front, left
            await scan_table.Move(170, 5)   # rear, left
            await scan_table.Move(5, 5)   # rear, right
        except Exception as ex:
            break_position = await scan_table.GetPosition()
            await scan_table.Home()
            step_error = scan_table.CoreXY.LastHomeStepErrors
            scan_table.ReloadConfig()
            msg = f'acc/dec: {a}; step_error:{step_error} stopped: {break_position[0]:.3f}, {break_position[1]:.3f}'
            await send_msg(json.dumps({"result": msg}))
            msg = f"{error_msg}{TEST_NAME}: Failed: {ex}"
            return(msg)
                
        await scan_table.Home()
        step_error = scan_table.CoreXY.LastHomeStepErrors
        msg = f'acc/dec: {a}; step_error:{step_error}'
        await send_msg(json.dumps({"result": msg}))

    scan_table.ReloadConfig()
    msg = f"{complete_msg}{TEST_NAME}: OK"
    logger.info(msg)
    return(msg)
